
# Standard libraries
import io

# Add-on modules
import dash
from dash import dcc
from dash_extensions.javascript import assign
from dash_iconify import DashIconify
import dash_leaflet as dl
import dash_mantine_components as dmc
import geopandas as gpd
import geojson
import numpy as np
import pandas as pd
import plotly.express as px
from shapely.geometry import Point
from shapely.geometry.polygon import Polygon
import plotly.graph_objects as go

# Our modules
import vis.util


class Effort:
    # empty data frame for when no data are available
    empty_frame = pd.DataFrame(
        data={'SpeciesId': ['None'], 'Effort': 'no effort'})

    s_per_day = 3600 * 24  # seconds in one day

    # Notifications - Dictionaries used with sendNotififcations events
    # in dmc.Notification objects.

    notify_duration_ms = 4000  # how long to display (ms)

    # User didn't select any effort on map/calendar
    no_effort_selected = dict(
        id="no_effort_selected",
        color="red",
        title="No effort selected",
        message="Selection does not include any effort",
        action="show",
        autoClose=notify_duration_ms
    )

    # User selected effort, but there were no detections in the effort
    no_associated_detections = dict(
        id="no_effort_associated_detections",
        color="red",
        title="No detections",
        message="No detections in selected effort",
        action="show",
        autoClose=notify_duration_ms
    )


    def __init__(self, explorer, debug=True):

        self.debug = debug
        self.explorer = explorer  # retain to run queries

        self.effort_df = self.get_effort()  # Find detection effort

        if len(self.effort_df) > 0:
            # Map species identifiers to human-readable names
            self.tsn_map = self.build_tsnmap(self.effort_df)
            for lang in self.explorer.languages:
                # Create Latin/vernacular names if effort data frame
                self.effort_df[lang] = [self.tsn_map.loc[tsn, lang]
                                        for tsn in self.effort_df["SpeciesId"]]

            # Query track information for mobile deployments
            self.tracks_by_effort = \
                self.get_mobile_deployment_tracks(self.effort_df)
            # Update mobile deployment lat/long to center of track
            for effId in self.tracks_by_effort.keys():
                latlon = self.tracks_by_effort[effId]
                if len(latlon.shape) > 1:
                    # Multiple rows, take the mean.  Note that we unwrap the longitude
                    # to take care of when we cross the antimeridian
                    lon = np.unwrap(self.tracks_by_effort[effId].Longitude, period=360)
                    lon = np.mean(lon)
                    lat = self.tracks_by_effort[effId].Latitude.mean()
                select_idx = self.effort_df.Id == effId
                self.effort_df.loc[select_idx, 'Latitude'] = lat
                self.effort_df.loc[select_idx, 'Longitude'] = lon

            # Create unfiltered copy of effort for data tables and maps
            self.selected_effort_df = self.effort_df  # data table
            self.selected_effort_Ids = self.effort_df # map
        else:
            # No effort data
            self.tsn_map = pd.DataFrame()
            self.tracks_by_effort = dict()

        self.detections = None  # no detection object yet

        self.point_to_layer = assign("""
                function(feature, latlng, context) {
                    /*
                     * Point to layer - Given a feature, a latitude/longitude pair,
                     * and a context object, render a marker for a point on the map.
                     * Size/color is dependent upon properties in the context hideout
                     * object and properties of the feature.  Icons are rendered with
                     * the number of days of recording effort.
                     */

                    const {min, max, colorscale, circleOptions, colorProp} = context.props.hideout;
                    const csc = chroma.scale(colorscale).domain([min, max]);  // chroma lib to construct colorscale
                    circleOptions.fillColor = csc(feature.properties[colorProp]);  // set color based on color prop.
                    // var marker = L.circleMarker(latlng, circleOptions);  // sender a simple circle marker.
                    const icon = L.divIcon.scatter({
                        html: '<div style="background-color:silver;"><span>' + Math.round(feature.properties.duration_d) + '</span></div>',
                        className: "marker-cluster",  // not really a marker cluster
                        iconSize: L.point(40, 40),
                        color: csc(feature.properties.duration_d)  // color code based on days of effort
                    })
                    
                    return L.marker(latlng, {icon : icon})
                }""")

        self.cluster_to_layer = assign("""
            function(feature, latlng, index, context) {
                /*
                 * cluster_to_layer
                 * Given a feature, latitude/longitude, cluster index, and context,
                 * render a cluster marker on the map.  We denote the number of
                 * days of recording effort as the sum of all of days of effort
                 * in all of the leaves associated with the marker.  As these are
                 * always displayed as integers, there may be slight discrepencies
                 * between the sum of days displayed in the children versus that
                 * of cluster set.
                 */
                 
                const {min, max, colorscale, circleOptions, colorProp} = context.props.hideout;
                console.log(min)
                const csc = chroma.scale(colorscale).domain([min, max]);            
                const leaves = index.getLeaves(feature.properties.cluster_id, limit=Infinity);     
                let valueSum = 0;
                for (let i = 0; i < leaves.length; ++i) {
                    // check if leaf is visible
                   
                    // extract out the start/end year
                    let start_y = parseInt(leaves[i].properties.start.match("[^-]*")[0]);
                    let end_y = parseInt(leaves[i].properties.end.match("[^-]*")[0]);
                    
                    // check for overlap
                    let latest_start = Math.max(start_y, context.props.hideout.y_filter_min);
                    let earliest_end = Math.min(end_y, context.props.hideout.y_filter_max);
                    let overlappedP = earliest_end - latest_start >= 0;
                    // check if the cluster is overlapping with the leaflet time range set
                    if (overlappedP) {               
                        valueSum += leaves[i].properties.duration_d
                    }                                              
                }                               

                let retval = null;
                if (valueSum > 0) {
                    // Render a circle with the total number of recording days in center
                    const icon = L.divIcon.scatter({
                        html: '<div style="background-color:white;"><span>' + Math.round(valueSum) + '</span></div>',
                        className: "marker-cluster",
                        iconSize: L.point(45, 45),
                        color: csc(valueSum)  // color code based on days of effort
                    });
                    retval = L.marker(latlng, {icon : icon});
                }
              
                return retval;
            }""")

        # Colors for intensity scale
        self.colorscale = ['red', 'yellow', 'green', 'blue', 'purple']

        self.geo_filter = assign("""
            function(feature, context) {
              /*
               * Determine whether or not a GEOJSON feature point should
               * be rendered based on the context.  Uses the hideout property
               * of the context and information about the feature
               */

               if ((feature.properies == undefined) || (feature.properties.hasOwnProperty('cluster') && feature.properties.cluster == true)) {
                   return true;
               } else { 
                   // extract out the start/end year            
                   let start_y = parseInt(feature.properties.start.match("[^-]*")[0]);
                   let end_y = parseInt(feature.properties.end.match("[^-]*")[0]);

                   // Check for overlap between deployment & display intervals
                   let latest_start = Math.max(start_y, context.props.hideout.y_filter_min);
                   let earliest_end = Math.min(end_y, context.props.hideout.y_filter_max);
                   let overlappedP = earliest_end - latest_start >= 0;               

                   return overlappedP;
               }
            }
            """)

    def build_tsnmap(self, df):
        """
        build_tsnmap - Given a dataframe with taxonomic serial numbers (TSNs),
        construct a dataframe that maps TSNs to human-readable names.
        The dataframe is indexed by TSN.
        :param df:  a data frame with TSNs in column SpeciesId
        :return:  dataframe with columns for each language for which we
          have species names.  Data from the Integrated Taxonomic Information
          System (www.itis.gov) contains entries for Latin, Enlgish, Spanish,
          and French, with the last two language only being partially complete.
          When values are missing, the Latin name is used.
        """
        # Get the taxonomic serial numbers (TSNs) in these data
        tsns = df['SpeciesId'].unique()
        if len(tsns) > 1:
            tsn_list = "(" + ", ".join([f"{tsn}" for tsn in tsns]) + ")"
        else:
            tsn_list = tsns[0]

        # Read the ITIS table, convert to dataframe,
        xml = self.explorer.xquery(self.explorer.xquery_byname('TSN') % (tsn_list))
        tsn_map = pd.read_xml(io.StringIO(xml), xpath="/ty:Result/ty:entry",
                                namespaces=self.explorer.namespaces)
        tsn_map.set_index("tsn", inplace=True)  # index by TSN

        # Handle missing data
        for lang in self.explorer.languages:
            if lang != "Latin":
                # replace missing vernacular entries with Latin names
                tsn_map.fillna({lang: tsn_map["Latin"]}, inplace=True)

        return tsn_map

    def get_effort(self):
        """
        Query effort
        :return: DataFrame with information about detection effort
        """

        # query for detection effort
        xml = self.explorer.xquery_byname('Effort', execute=True)
        # Verify that we received a result (true except in case of empty database)
        match = vis.multiple_elem_re.search(xml[0:vis.multiple_elem_N])
        if match:
            # Parse data into a frame, flattening with an XLST processing script
            effort_df = pd.read_xml(io.StringIO(xml),
                                    stylesheet=self.explorer.xslt["Effort"],
                                    namespaces=self.explorer.namespaces
                                    )
            for tfield in ("Start", "End"):
                effort_df[tfield] = pd.to_datetime(
                    effort_df[tfield], utc=True)
            vis.util.longitude_range_pm180(effort_df)

            # compute duration and marker sizes (markers log scale w/ minimal value)
            min_s = 3600 / 2  # 30 min
            effort_df['Duration'] = effort_df['End'] - effort_df['Start']
            effort_df['MarkerSize'] = \
                np.log(effort_df.Duration.dt.total_seconds().clip(lower=min_s))
        else:
            effort_df = pd.DataFrame()  # empty dataframe

        return effort_df

    def get_mobile_deployment_tracks(self, effort_df):
        """
        get_mobile_deployment_tracks
        Given an effort dataframe with a Track column indicating GPS track
        data, query all deployments associated with the efforts to retrieve
        GPS information.  Currently, we query the entire track line even if
        it might fall outside of the effort.

        :param effort_df: A dataframe of effort
        :return: pair of dictionaries:
            gps series keyed by deployment identifier
            gps series keyed by effort identifier
        """
        effort_mobile = effort_df[effort_df.Track == True]
        eff_tracks = {}  # Tracks by effort
        eff_by_dep = effort_mobile.groupby("DeploymentId")
        for dep, df in self.explorer.deployments.tracks_by_deployment.items():
            if dep in eff_by_dep.groups.keys():
                # There is effort in this deployment, extract sub-tracks
                # for each effort associated with the deployment.
                depeff_df = eff_by_dep.get_group(dep)
                effort_groups = depeff_df.groupby("Id")
                for eff, eff_df in effort_groups:
                    # Any row associated with this effort will have Start/End
                    # use the first one
                    row = eff_df.iloc[0]

                    # Find the first and last rows in the deployment
                    # that are associated with this effort.
                    first = df.index.searchsorted(row.Start, side="right")
                    last = df.index.searchsorted(row.End, side="left")
                    if len(df) >= first:
                           assert True, f"Detection effort {row.Id} is a mobile deployment, but there are " + \
                                        f"no GPS tracks between {row.Start} and {row.End}."
                    else:
                        if first == last:
                            # We are off the track or the resolution of the sampling
                            # is poor.  Just use one sample
                            eff_tracks[eff] = df.iloc[first]
                        else:
                            eff_tracks[eff] = df.iloc[first:last]

        return eff_tracks

    def getPointsInPolygon(self, seldata, effort_fig_bounds):
        """
        Retrieve effort within a user selection
        :param seldata:  Leaflet callback data representing a user selection
        :param effort_fig_bounds:  Map viewport bounds, only used with markers
        :return:
        """

        if len(seldata['features']) == 0:
            return []
        else:
            coordinates = seldata['features'][0]['geometry']['coordinates']
            shape = seldata['features'][0]['properties']['type']
            # Set geodetic coordinate system
            WGS84 = 4326  # EPSG:4326 (WGS 84)
            crs_m = 3857  # geodetic coordinate system in meters

            if shape == 'rectangle' or shape == 'polygon':
                coordinates = coordinates[0]
            elif shape == 'marker':
                # get the scope of the world from the level of zoom
                bound1 = Point(effort_fig_bounds[0])
                bound2 = Point(effort_fig_bounds[1])

                # does not work
                # https://stackoverflow.com/questions/63722124/get-distance-between-two-points-in-geopandas
                points_df = gpd.GeoDataFrame({'geometry': [bound1, bound2]}, crs=WGS84)
                points_df = points_df.to_crs('EPSG:5234')
                # We shift the dataframe by 1 to align pnt1 with pnt2
                points_df2 = points_df.shift()
                dist_of_map = points_df.distance(points_df2)

                points = self.selected_effort_df[
                    (self.selected_effort_df['Longitude'] == coordinates[0]) &
                    (self.selected_effort_df['Latitude'] == coordinates[1])]
                return points

            elif shape == 'circle':
                # Create WGS 84 point
                gdf = gpd.points_from_xy((coordinates[0],), (coordinates[1],), crs=WGS84)
                # Transform to meters
                gdf_m = gdf.to_crs(crs=crs_m)
                # Create buffer in meters around point
                radius_m = seldata['features'][0]['properties']['_mRadius']
                buffer_m = gdf_m.buffer(radius_m)
                buffer_wgs84 = buffer_m.to_crs(crs=WGS84)
                # Extract coordinates
                xyCoords = buffer_wgs84.data[0].boundary.xy
                coordinates = list(zip(xyCoords[0], xyCoords[1]))
                # change from list of tuples to list of lists for coordinates
                coordinates = [list(x) for x in coordinates]

            polygon = Polygon(coordinates)
            # get long lat out of df and see if its in polygon
            boolCoorInPolygon = [polygon.contains(Point(x)) for x in self.selected_effort_df[['Longitude', 'Latitude']].values]
            # filter to get points in polygon
            points = self.selected_effort_df[boolCoorInPolygon]

            return points

    def select_on_timeline(self, sel_data, language):
        """
        :param sel_data:  selection event on timeline plot
        :param language: Selected language on settings tab
        :return:  list of state updates:
           detections summary table data and map summary table data which helps to filter data on detections tab
           detections table column names
           currently viewed tab
           notification if appropriate and
           detections_hr_week_fig, detections_diel_fig, detections_map_fig
        """

        if sel_data is None:
            # System triggers some callbacks w/ no selections
            raise dash.exceptions.PreventUpdate

        if not isinstance(sel_data, dict)  or \
                ('points' in sel_data and len(sel_data['points']) == 0):
            return dash.no_update, dash.no_update, dash.no_update, dash.no_update, \
                'effort', self.no_effort_selected, dash.no_update, dash.no_update, dash.no_update, dash.no_update
        else:
            targets_sel = [curve['pointIndex'] for curve in sel_data['points']]
            if self.debug and len(targets_sel) > 0:
                print("selection data")
                print(self.selected_effort_df.iloc[targets_sel])
                print()
            targets = targets_sel

        # Create subset of effort table and query detections
        efforts = self.selected_effort_df.iloc[targets_sel]
        detections = self.get_detections_from_efforts(efforts)

        if detections is self.empty_frame:
            # no detections found, leave detections table as it was,
            # remain on the effort tab and inform the user of their sorry
            # luck
            return dash.no_update, dash.no_update, dash.no_update, dash.no_update, \
                'effort', self.no_associated_detections, dash.no_update, dash.no_update, dash.no_update, dash.no_update

        # Set detection data
        self.detections.set_data(efforts, self.tsn_map, detections)

        # Return values for tabular representation
        det_dict, det_cols = self.detections.table(language)

        # adding overlap column to the map summary datatable as indicator that is record visible over map or not
        det_dict, map_det_cols = Effort.add_overlap_column_map_summary_tbl(det_cols, det_dict)

        # Return new detections table and transition to the detections tab
        return det_dict, det_cols, det_dict, map_det_cols, 'detections', [], go.Figure(), {}, go.Figure(), go.Figure()

    def select_on_map(self, sel_data, effort_fig_bounds, language):
        """
        selection - Find detections based on effort
        :param sel_data:  Callback dictionary indicating the region
           and points that the user has selected
        :param effort_fig_bounds:  Callback dictionary indicating the region
           and points that the user has selected
        :param language: Selected language on settings tab
        :return:  list of state updates:
           detections summary table data and map summary table data which helps to filter data on detections tab
           detections table column names
           leaflet edit control action
           currently viewed tab
           notification if appropriate and
           detections_hr_week_fig, detections_diel_fig, detections_map_fig
        """

        timestamp = pd.Timestamp.now()
        print(f"\nselection at {timestamp}:  ")
        errors = None

        rm_selection = dict(mode="remove", action="clear all")
        # The selection and click data will have custom data whose
        # first element is an index into the effort data frame.

        efforts = None

        if sel_data is not None and len(sel_data['features']) > 0:
            efforts = self.getPointsInPolygon(sel_data, effort_fig_bounds)
            if len(efforts) == 0:
                # Don't update detections table, remove highlighting
                # of user selection, stay on the effort tab and provide
                # feedback that we didn't select anything.
                return dash.no_update, dash.no_update, dash.no_update, dash.no_update, rm_selection, \
                    'effort', self.no_effort_selected, dash.no_update, dash.no_update, dash.no_update, dash.no_update
        else:
            # Nothing at all selected, might be a callback due to removing
            # a leaflet selection
            raise dash.exceptions.PreventUpdate

        # Something was selected, no need to notify
        notify = []

        detections = self.get_detections_from_efforts(efforts)

        if detections is self.empty_frame:
            # no detections found, leave detections table as it was,
            # remove the selection area, remain on the effort
            # tab and inform the user
            return dash.no_update, dash.no_update, dash.no_update, dash.no_update, rm_selection, \
                'effort', self.no_associated_detections, dash.no_update, dash.no_update, dash.no_update, dash.no_update

        # Set detection data
        self.detections.set_data(efforts, self.tsn_map, detections)
        # Return values for tabular representation
        det_dict, det_cols = self.detections.table(language)

        # adding overlap column to the map summary datatable as indicator that is record visible over map or not
        det_dict, map_det_cols = Effort.add_overlap_column_map_summary_tbl(det_cols, det_dict)

        # Return new detections table, shut off the selection,
        # and move to the detections tab
        return det_dict, det_cols, det_dict, map_det_cols, rm_selection, 'detections', [], go.Figure(), {}, go.Figure(), go.Figure()

    @classmethod
    def add_overlap_column_map_summary_tbl(cls, det_cols, det_dict):
        """
        adding overlap column to the map summary datatable as indicator that is record visible over map or not
        :param det_cols: detections summary table columns
        :param det_dict: detections summary table data records
        :return: copy of summary table column and data with more column that indicates is record visible over map or not
        """
        overlap_col = [["True"] for i in range(len(det_dict))]
        map_det_cols = det_cols.copy()
        map_det_cols.append({'id': 7, 'name': 'Displayed on Map'})
        det_dict = np.append(det_dict, overlap_col, axis=1)
        return det_dict, map_det_cols

    def get_detections_from_efforts(self, efforts):
        """
        Given rows from an effort table, query all detections associated
        with the effort
        :param efforts: sub-dataframe of effort to query
        :return: detections dataframe
        """

        query_elements = [
            'declare default element namespace "http://tethys.sdsu.edu/schema/1.0";',
            '<Result xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">'
        ]

        # Iterate over query documents, building criteria
        effort_groups = efforts.groupby("Id")
        for eff_name, eff_grp in effort_groups:
            conditions = []
            has_subtype = "Subtype" in eff_grp.keys()
            # We do not check Effort section as we know that all selections
            # meet the effort criteria
            detections_selection = f'Id = "{eff_name}"'
            if len(eff_grp) > 0:
                for _idx, row in eff_grp.iterrows():
                    cond_grp = []
                    cond_grp.append(f'SpeciesId = {row.SpeciesId}')

                    if row.Call is not None:
                        cond_grp.append(f'Call = "{row.Call}"')
                    if has_subtype and row.Subtype is not None:
                        cond_grp.append(f'Parameters/Subtype = "{row.Subtype}"')
                    conditions.append(" and ".join(cond_grp))
                det_condition = f"({') or ('.join(conditions)})"
                # Format the query string and add it to the query
                query_elements.append(
                    self.explorer.xqueries['DetectionsFromEffort']%(
                        detections_selection, det_condition))
        # Close off the query
        query_elements.append("</Result>")

        xquery = "\n".join(query_elements)
        xml = self.explorer.xquery(xquery)

        updated = False
        try:
            detections = pd.read_xml(
                xml, xpath="/ns0:Result/ns0:Detection",
                namespaces={"ns0": "http://tethys.sdsu.edu/schema/1.0"},
                parse_dates=["Start"]
            )
            # Detection end is optional and will not be in all columns.
            # read_xml (as of 1.5.3) will not convert any values if any one
            # of them do not parse correctly.  Use to_datetime which will
            # convert to NaT values when absent
            if "End" in detections.columns:
                detections["End"] = pd.to_datetime(detections["End"], utc=True)

        except ValueError as v:
            if self.debug:
                print("No detections.  Query & Result:\n--- xquery ---\n")
                print(xquery)
                print("\n--- result ---\n")
                print(xml)
                print()
            detections = self.empty_frame

        return detections

    def plot(self, effort_display, granularity, language):
        """
        plot
        :param effort_display:  Show as map | calendar
        :param granularity:  Report effort at which granularity
        :param language:  Taxonomy representation (Latin, English, French, Spanish)
        :param tsn: SpeciesId from filter
        :return:  plotly express figure handle

        Caveats:  This is a callback function and should not be called directly
        """

        df = self.effort_df.copy()
        # Create a Species/Call column and sort on it
        has_groups = "Group" in df.keys()
        form_SpeciesCallLabel = lambda df, idx: \
           df.at[idx, language] + \
           (df.at[idx, "Group"]
              if has_groups and df.at[idx, "Group"] else "") + \
           f'/{df.at[idx, "Call"]}' if df.at[idx, "Call"] is not None else ""
        df["Species/Call"] = [form_SpeciesCallLabel(df, idx)
                              for idx in range(len(self.effort_df))]
        df.loc[:, 'days'] = df["Duration"].dt.days
        # Sort by TSN then call
        sorted_df = df.sort_values(
            by=['SpeciesId', 'Call'], ascending=True,
            key=lambda y: y.lower() if isinstance(y, str) else y)

        selected_df = sorted_df[(sorted_df["Granularity"] == granularity)]
        # # default species id until we get query
        # if species_dropdown is not None:
        #     species_dropdown = list(map(int, species_dropdown))
        #
        # selected_df = sorted_df[(sorted_df["Granularity"] == granularity)
        #                         & ((species_dropdown is None)
        #                            or (len(list(set(species_dropdown) & set(sorted_df['SpeciesId']))) > 0))]

        # selected_df = sorted_df[sorted_df["Granularity"] == granularity]
        if effort_display == "calendar":
            fig = px.timeline(
                selected_df,
                x_start="Start", x_end="End", y="Species/Call",
                custom_data=[selected_df.index.values.tolist()],
                hover_data=["Species/Call",
                    "Start", "End", "days", "Method", "Software", "Version",
                    "Id", "Longitude", "Latitude"],
                category_orders={
                    "column":sorted(selected_df["Species/Call"], key=str.casefold)},
            )

        elif effort_display == "map":
            zoom, center = vis.util.zoom_center(lons=selected_df.Longitude,
                                                lats=selected_df.Latitude)
            fig = px.scatter_mapbox(
                selected_df,
                custom_data=[selected_df.index.values.tolist()],
                lat='Latitude', lon='Longitude', size_max=9, size='MarkerSize',
                opacity=0.6,
                color="Species/Call",
                color_continuous_scale="rainbow",
                hover_data={"MarkerSize": False, "Latitude": False, "Longitude": False,
                            'SpeciesId': True, 'Call': True, 'Start': True,
                            'End': True, 'Duration': True,
                            'Id': True, 'DeploymentId': True},
                zoom=zoom,
                center=center
                )

            #fig.data[0].update(hovertemplate='Species/Call=%{marker.SpeciesId}/%{marker.Call}<br>peak_hour=%{marker.color}')

            # fig.update_traces(cluster_enabled=True, selector=dict(type='scattermapbox'))
            # fig.update_traces(cluster_step=2, selector = dict(type='scattermapbox'))
            # fig.update_traces(patch={"cluster": { "enabled":True } })
            # fig.show()

            shape_dict = self.explorer.get_shapes()
            layers = list()
            for name, geo_df in shape_dict.items():
                # Identify fields with strings
                str_indices = geo_df.dtypes[geo_df.dtypes == 'object'].index
                if len(str_indices) == 0:
                    print('here')
                    continue
                    # No names are associate with this shape, use the filename
                    geo_df['Name'] = name
                    str_indices = ['Name']
                # Each asset might have multiple items
                # choropleth = px.choropleth_mapbox(
                #     data_frame=geo_df,
                #     geojson=geo_df.geometry,
                #     locations=geo_df.index,
                #     opacity=0.15,
                #     # We make the assumption that the first column to
                #     # use a string contains the name
                #     color=str_indices[0])
                # fig.add_trace(list(choropleth.select_traces()))
                #plots.append(map)
                layers.append({
                    'source': geo_df.__geo_interface__,
                    #'sourcetype':'geojson',
                    'below': 'traces',
                    'color': 'pink',
                    'type': 'line',
                    'opacity': 0.15
                })


            # Show hover for all points at same lat (x) or long (y)

            fig.update_layout(
                hovermode="x",
                mapbox={
                    "style": "mapbox://styles/mroch/cl1z3qvva003k14oj2ymnxamu",
                    # Add in any user-specified shapes (see shapes.Shapes)
                    'layers': layers
                    # "layers": [{
                    #     "sourcetype": "geojson",
                    #     "source": assets[0],
                    #     "below": "",
                    #     "type": "fill",
                    #     "color": "orange",
                    #     "line": {"width": 2.5}
                    # }]
                },
            )
        fig.update_layout(clickmode='event+select')
        #fig.update_traces(cluster_size=2)

        self.fig = fig  # debug:  keep so we can inspect on callback

        #create speciesId dropdown options
        speciesOptions = {str(val): str(val) for val in list(selected_df['SpeciesId'].unique())}

        # Return new figure
        return fig, speciesOptions

    def effort_table_filter(self, granularity, algorithm_filter, language,
                            species_filter_N=None,
                            species_filter_id=None,
                            species_selected=None):
        """
        :param granularity:  effort granularity
        :param algorithm_filter:  effort algorithm filter
        :param language: Selected language on settings tab
        :param species_filter_N:  counter
        :param species_filter_id:  Array of dict describing potential species
        :param species_selected: Array showing which species filters are selected
        :return:  Plotly DataTable values, DataTable columns
        """

        # See if we need to cascade selections in the species filter
        trigger = dash.ctx.triggered_id

        # Determine how taxa are being filtered regardless of whether
        # or not other filters still have remaining items as we want
        # to update the number of tax in search message
        tsns = []
        if species_selected is not None and any(species_selected):
            # Build taxonomic serial number list for species of interest
            for selected, what in zip(species_selected, species_filter_id):
                if selected and what['index'] != "":
                    tsns.append(int(what['index']))
        # Provide an indication of whether the species filters are active
        filter_label = f"Filter ({len(tsns)} ranks)" if len(tsns) else "Filter"

        # Handle empty effort case
        if len(self.effort_df) == 0:
            tbl_vals, tbl_cols = vis.util.dataframe_to_table(self.effort_df)
            return tbl_vals, tbl_cols, filter_label

        # Build true/false indicator array to filter effort
        # granularity
        select = self.effort_df['Granularity'] == granularity
        if any(select):
            if algorithm_filter is not None and len(algorithm_filter) > 0:
                # algorithms with specified granularity
                select = select & \
                    self.effort_df['Software'].isin(algorithm_filter)

        if len(tsns):
            select = select & \
                 self.effort_df['SpeciesId'].isin(tsns)

        # Use indicator array to determine restricted effort
        self.selected_effort_df = self.effort_df[select.values]

        # Set up plotly DataTable
        # Use current language for species and set duration in days
        tbl_vals, tbl_cols = vis.util.dataframe_to_table(
            self.selected_effort_df,
            ['Id', 'Project', 'Site', 'Cruise', language, 'Track', 'Start', 'Duration',
             'Software', 'RecordingDuration_s', 'RecordingInterval_s'],
            False
        )

        # Set up reduced effort table for maps
        # Reduce to single entry for each effort document
        # We may still have multiple effort documents covering the same time
        # span...
        self.selected_effort_Ids = self.selected_effort_df.drop_duplicates("Id")
        ids = self.selected_effort_df.Id.unique()
        self.selected_effort_Ids = self.selected_effort_Ids[
            ["Id", "DeploymentId", "Project", "Site", "Track", "Start", "End",
             "Longitude", "Latitude", "Duration"]]

        return tbl_vals, tbl_cols, filter_label

    def effort_timeline_graph(self, effort_tbl, language):
        """
        Produce figure children to display a timeline chart of effort.
        Timeline chart shows species and call effort
        :param language: Selected language on settings tab
        :param effort_tbl:  effort table.  These data are only used as a
           callback dependency trigger
        :return:  list of dcc.Graph children (usually contains one px.timeline)
        """

        # Create a version of the selected effort with a species/call column
        effort_df = self.selected_effort_df
        if len(effort_df) > 0:
            ycol = effort_df.loc[:, [language, 'Call']].agg(lambda x: '/'.join(x) if x[1] is not None else x[0], axis=1)
            # Draw the timeline chart
            timeline = px.timeline(
                effort_df,
                x_start="Start", x_end="End", y=ycol,
                # custom_data=[effortDf.index.values.tolist()],
                hover_data=[language, "Call",
                            "Start", "End", "Duration", "Method", "Software", "Version",
                            "Id", "Longitude", "Latitude"],
                category_orders={
                    "column": sorted(effort_df, key=lambda x: (x[0], x[1]))},
                labels={"y": "Species/Call"}
            )
            # toolbar default select data
            timeline.update_layout(dragmode="select")
        else:
            timeline = dict()

        return timeline

    def effort_map(self, effort_tbl):
        """
        effort_map - Populate an effort map
        :param effort_tbl:  Dummy variable, data passed in when effort table
           is updated so that we draw a new map.
        :return: map
        """

        effortDf = self.selected_effort_Ids

        # Group rows of effort by effort Id

        if len(effortDf) == 0:
            # No data, return empty features
            hideout = dict(
                min=0,  # no data
                max=1,
                circleOptions=dict(fillOpacity=1, stroke=False, radius=10),
                colorProp="density",
                colorscale=self.colorscale,
                y_filter_min=2000,  # dummy values
                y_filter_max=2001,
            )
            stationary = None
            mobile = None
            return hideout['max'], stationary, mobile, hideout, hideout

        trackI = effortDf.Track == True
        mobile = effortDf.loc[trackI, :]
        fixed = effortDf.loc[~trackI, :]

        # Add mobile deployments
        mobile_ids = mobile.Id
        line_features = []
        for id, row in mobile.iterrows():
            track = self.explorer.deployments.tracks_by_deployment[row.DeploymentId]
            # Unwrap any longitudes that cross the anti-meridian
            new_track = vis.util.longitude_unwrap(track)
            # Put in long/lat tuple order
            point_set = [tuple(t) for t in new_track.values]

            point_set = geojson.LineString(point_set)
            start = row.Start.strftime('%Y-%m-%d')
            duration_d = (row.End - row.Start).days
            tooltip_str = f'{row.Id}: {start}, {duration_d:.1f} days'
            feature = geojson.Feature(
                geometry=point_set,
                properties={
                    'name': tooltip_str,
                    'tooltip': tooltip_str,
                    'start': row.Start.strftime('%Y-%m-%d %XZ'),
                    'end': row.End.strftime('%Y-%m-%d %XZ'),
                    'duration_d': duration_d
                }

            )
            line_features.append(feature)

        points = []
        for idx, row in fixed.iterrows():
            # GeoJSON points are easting (-180,180], northing [-90,90]
            point = geojson.Point((row.Longitude, row.Latitude))
            start = row.Start.strftime('%Y-%m-%d')

            # recovery field is not mandatory
            try:
                endtime = row.End.strftime('%Y-%m-%d %XZ')
                # Always have at least one day of effort or display fails
                duration_d = max(1, (row.End - row.Start).round('d').days)
                tooltip_str = f'{row.Id}: {start}, {duration_d} days'
            except ValueError as e:
                endtime = "NaT"
                tooltip_str = f'{row.Id}: {start}, no recovery field'

            geo = geojson.Feature(
                geometry=point,
                properties={
                    'name': tooltip_str,
                    'tooltip': tooltip_str,
                    'start': row.Start.strftime('%Y-%m-%d %XZ'),
                    'end': endtime,
                    'duration_d': duration_d
                }
            )
            points.append(geo)

        # Set up additional information for JavaScript client in a "hideout"
        # data structure

        # When we cluster items, the maximum duration becomes longer.
        # Select an amplification factor for the longest duration recording
        # to account for clusters of recordings.  Large clusters will likely
        # be off the color scale
        amplifyN = 1.25
        # Compute effort in days (might overflow with default ns)
        effort_days = effortDf.Duration.astype('timedelta64[s]').sum()/self.s_per_day

        # set defaults for timespan to be shown:
        # earliest deployment and latest recovery
        earliest = effortDf.Start.min().year
        latest = effortDf.End.max().year

        hideout = dict(
            min=effortDf.Duration.min().days,
            max=effort_days*amplifyN,
            circleOptions=dict(fillOpacity=1, stroke=False, radius=10),
            colorProp="density",
            colorscale=self.colorscale,
            y_filter_min=earliest,
            y_filter_max=latest,
        )

        stationary = geojson.FeatureCollection(points)
        mobile = geojson.FeatureCollection(line_features)

        return hideout['max'], stationary, mobile, hideout, hideout

    def effort_map_leaflet(self):
        colorbar = dl.Colorbar(id="effort_map_colorbar",colorscale=self.colorscale, width=20, height=150,
                               min=0, unit='days')
        tile_layer = dl.TileLayer(id="deteffort_tilelayer")
        children = [
                tile_layer,
                dl.FeatureGroup([
                    dl.EditControl(id="effort_selection_control",
                                   draw={
                                       'polyline': 'false',
                                       'polygon': 'false',
                                       'circle': 'False',
                                       'rectangle': {
                                           'shapeOptions': {
                                               'clickable': 'false'
                                           }
                                       },
                                   }
                                   )
                ]),
                # Cluster markers.  Show in spider formation if too many
                # on map when zoomed all the way in, and allow clicks to
                # zoom in
                dl.GeoJSON(
                    id="stationary_efforts",
                    cluster=True,
                    spiderfyOnMaxZoom=True,
                    zoomToBoundsOnClick=True,
                    clusterToLayer=self.cluster_to_layer,
                    options=dict(pointToLayer=self.point_to_layer,
                                 filter=self.geo_filter)
                ),
                dl.GeoJSON(
                    id="mobile_effort",
                    options=dict(filter=self.geo_filter),
                ),
                colorbar,
            ]

        # Plot Marine Sanctuary Geojson
        shape_dict = self.explorer.get_shapes()
        for name, geo_df in shape_dict.items():
            children.append(dl.GeoJSON(
                data=geo_df.__geo_interface__,
                # click_feature=None,
                zoomToBoundsOnClick=False,
                id=f'{name}',
                options={"style": {"color": "pink"}}
            ))

        leaflet = dl.Map(
            center=[32.37, -120],
            children=children,
            id="effort_map",
            style={'width': '100%', 'height': '50vh',
                   'margin': "auto", "display": "block"}
        )
        return leaflet

    def make_mirror(self):
        mirror = dl.Map(center=[56, 10], zoom=4, children=[
            dl.TileLayer(),
            dl.GeoJSON(id="geojson", options=dict(pointToLayer=self.point_to_layer), zoomToBounds=True),
        ], style={'width': '50%', 'height': '50vh', 'margin': "auto", "display": "inline-block"},
               id="mirror")
        return mirror

    def set_detections(self, detections):
        """
        set_detections - Provide detections object to which we will report
        effort selections
        :param detections:
        :return:
        """
        self.detections = detections


    def download_table(self, clicks, language):
        """
        Download the effort table
        Note that the summary/complete option currently has no effect and
        is not used.
        :param clicks:  # of times clicked (we don't care)
        :param language:  language for species id
        :return:
        """
        download_options = self.explorer.get_table_download_characteristics()
        # Make shallow copies of the data so that we can modify specific columns
        # without modifying the underlying data
        table = self.selected_effort_df.copy(deep=False)
        fname = "effort"

        # Rename SpeciesId to human readable data
        table.SpeciesId = \
            self.tsn_map.loc[table.SpeciesId, language].values

        buf = io.BytesIO()
        if download_options['format'] == 'Excel':
            # Remove timezone information (not supported by to_excel)
            for attr in ('Start', 'End'):
                table[attr] = table.loc[:, attr].dt.tz_localize(None)
            table.to_excel(buf, index=False)
            fname = fname + ".xlsx"
        else:
            # Produce comma seperated value list without row names
            table.to_csv(buf, index=False)
            fname = fname + ".csv"
        buf.seek(0)  # Rewind to start of buffer

        # Return the data to be downloaded
        return dcc.send_bytes(buf.getvalue(), fname)
